<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;
use App\Models\Scopes\TenantScope;


class WaitingQueue extends Model
{
    use HasFactory;

    protected $table = 'waiting_queue';

    protected $fillable = [
        'customer_id',
        'type',
        'appointment_time',
        'arrival_time',
        'status',
        'position',
        'priority_score',
        'estimated_wait_time',
        'notes',
        'served_by',
        'served_at',
        'completed_at',
        'added_by',
        'tenant_id',
        'visit_type',
    ];

    protected $casts = [
        'appointment_time' => 'datetime',
        'arrival_time' => 'datetime',
        'served_at' => 'datetime',
        'completed_at' => 'datetime'
    ];


    protected static function booted()
    {
        static::addGlobalScope(new TenantScope);

        static::creating(function ($model) {
            if (app()->bound('currentTenant') && empty($model->tenant_id)) {
                $model->tenant_id = app('currentTenant')->id;
            }
        });
    }

public function tenant()
{
    return $this->belongsTo(Tenant::class);
}

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            if (!$model->arrival_time) {
                $model->arrival_time = now();
            }
        });

    }

    public function customer()
    {
        return $this->belongsTo(Customer::class);
    }

    public function servedBy()
    {
        return $this->belongsTo(User::class, 'served_by');
    }

    public function addedBy()
    {
        return $this->belongsTo(User::class, 'added_by');
    }

       // ✅ كاست مع مراعاة التوقيت الخاص بالـ Tenant
       protected function asDateTime($value)
       {
           $date = parent::asDateTime($value);

           if (app()->bound('currentTenant') && app('currentTenant')->timezone) {
               return $date->clone()->timezone(app('currentTenant')->timezone);
           }

           return $date;
       }

    public function scopeWaiting($query)
    {
        return $query->where('status', 'waiting');
    }

    public function scopeAppointments($query)
    {
        return $query->where('type', 'appointment');
    }

    public function scopeWalkIns($query)
    {
        return $query->where('type', 'walk-in');
    }

    public function isAppointmentDue()
    {
        if ($this->type !== 'appointment' || !$this->appointment_time) {
            return false;
        }

        $now = Carbon::now(app()->bound('currentTenant')->timezone ?? config('app.timezone'));
        $appointmentTime = $this->appointment_time;

        // Consider appointment due if within 15 minutes
        return $appointmentTime <= $now;
    }

}
