<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;
use App\Models\Scopes\TenantScope;



class AppointmentSetting extends Model
{
    use HasFactory;

    protected $fillable = [
        'interval_minutes',
        'start_time',
        'end_time',
        'working_days',
        'max_appointments_per_slot',
        'allow_same_day_booking',
        'tenant_id',
    ];

    protected $casts = [
        'working_days' => 'array',
        'allow_same_day_booking' => 'boolean',
        'start_time' => 'datetime:H:i',
        'end_time' => 'datetime:H:i',
    ];

    protected static function booted()
    {
        static::addGlobalScope(new TenantScope);

        static::creating(function ($model) {
            if (app()->bound('currentTenant') && empty($model->tenant_id)) {
                $model->tenant_id = app('currentTenant')->id;
            }
        });
    }

    public static function getSettings()
    {
        return self::first() ?? self::create([]);
    }

    public function getAvailableTimeSlots($date = null)
    {
        $date = $date ?? now()->format('Y-m-d');
        $slots = [];

        $start = \Carbon\Carbon::parse($date . ' ' . Carbon::parse($this->start_time)->format('H:i:s'));
        $end = \Carbon\Carbon::parse($date . ' ' . Carbon::parse($this->end_time)->format('H:i:s'));

        while ($start->lt($end)) {
            $time = $start->format('H:i');

            if ($this->isTimeSlotAvailable($date, $time)) {
                $slots[] = $time;
            }

            $start->addMinutes($this->interval_minutes);
        }

        return $slots;
    }

    public function isTimeSlotAvailable($date, $time)
    {
        $existingAppointments = \App\Models\WaitingQueue::where('type', 'appointment')
            ->whereDate('appointment_time', $date)
            ->whereTime('appointment_time', $time)
            ->count();

        return $existingAppointments < $this->max_appointments_per_slot;
    }
}
