<?php

namespace App\Http\Controllers;

use App\Models\ShortLink;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class ShortLinkController extends Controller
{
    /**
     * Redirect to original URL using short code
     */
    public function redirect($shortCode)
    {
        $shortLink = ShortLink::where('short_code', $shortCode)
            ->active()
            ->first();
            
        if (!$shortLink) {
            abort(404, 'Short link not found or expired');
        }
        
        // Check if expired
        if ($shortLink->isExpired()) {
            abort(410, 'This link has expired');
        }
        
        // Record the click
        $shortLink->recordClick();
        
        // Log for analytics
        Log::info('Short link accessed', [
            'short_code' => $shortCode,
            'original_url' => $shortLink->original_url,
            'user_agent' => request()->userAgent(),
            'ip' => request()->ip(),
            'tenant_id' => $shortLink->tenant_id
        ]);
        
        // Redirect to original URL
        return redirect($shortLink->original_url);
    }
    
    /**
     * Show short link analytics
     */
    public function analytics($shortCode)
    {
        $shortLink = ShortLink::where('short_code', $shortCode)->first();
        
        if (!$shortLink) {
            abort(404, 'Short link not found');
        }
        
        // Check if user has permission to view analytics
        if (auth()->guest() || auth()->user()->tenant_id !== $shortLink->tenant_id) {
            abort(403, 'Unauthorized to view analytics');
        }
        
        return response()->json([
            'short_code' => $shortLink->short_code,
            'original_url' => $shortLink->original_url,
            'short_url' => $shortLink->short_url,
            'click_count' => $shortLink->click_count,
            'created_at' => $shortLink->created_at,
            'expires_at' => $shortLink->expires_at,
            'is_expired' => $shortLink->isExpired(),
            'usage_type' => $shortLink->usage_type,
            'metadata' => $shortLink->metadata
        ]);
    }
    
    /**
     * Create a new short link
     */
    public function create(Request $request)
    {
        $request->validate([
            'original_url' => 'required|url',
            'usage_type' => 'required|string|in:display_pairing,queue_public,general',
            'expires_at' => 'nullable|date|after:now',
            'metadata' => 'nullable|array'
        ]);
        
        $shortLink = ShortLink::create([
            'original_url' => $request->original_url,
            'short_code' => ShortLink::generateUniqueCode(),
            'tenant_id' => auth()->user()->tenant_id,
            'usage_type' => $request->usage_type,
            'expires_at' => $request->expires_at,
            'metadata' => $request->metadata,
            'created_by' => auth()->user()->email
        ]);
        
        return response()->json([
            'success' => true,
            'short_link' => [
                'id' => $shortLink->id,
                'short_code' => $shortLink->short_code,
                'short_url' => $shortLink->short_url,
                'original_url' => $shortLink->original_url,
                'expires_at' => $shortLink->expires_at
            ]
        ]);
    }
    
    /**
     * Regenerate short code for existing link
     */
    public function regenerate($id)
    {
        $shortLink = ShortLink::where('id', $id)
            ->where('tenant_id', auth()->user()->tenant_id)
            ->first();
            
        if (!$shortLink) {
            abort(404, 'Short link not found');
        }
        
        $shortLink->update([
            'short_code' => ShortLink::generateUniqueCode(),
            'click_count' => 0, // Reset click count
            'expires_at' => $shortLink->usage_type === 'display_pairing' ? now()->addMonths(6) : $shortLink->expires_at
        ]);
        
        return response()->json([
            'success' => true,
            'message' => 'Short link regenerated successfully',
            'short_link' => [
                'id' => $shortLink->id,
                'short_code' => $shortLink->short_code,
                'short_url' => $shortLink->short_url,
                'original_url' => $shortLink->original_url
            ]
        ]);
    }
    
    /**
     * Delete a short link
     */
    public function delete($id)
    {
        $shortLink = ShortLink::where('id', $id)
            ->where('tenant_id', auth()->user()->tenant_id)
            ->first();
            
        if (!$shortLink) {
            abort(404, 'Short link not found');
        }
        
        $shortLink->delete();
        
        return response()->json([
            'success' => true,
            'message' => 'Short link deleted successfully'
        ]);
    }
}
