<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Customer;
use App\Models\WaitingQueue;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use App\Models\ActivityLog;
use App\Models\VisitType;
use App\Events\QueueUpdated;

class CustomerController extends Controller
{
    public function __construct()
    {
        $this->middleware("auth");
    }

    public function index(Request $request)
    {
        $search = $request->input('search');

        $customers = Customer::when($search, function ($query, $search) {
                return $query->where('name', 'like', "%$search%")
                             ->orWhere('phone', 'like', "%$search%");
            })
            ->orderBy('created_at', 'desc')
            ->get();


            return view('customers.index', compact('customers', 'search'));
    }

    public function create()
    {

        $visitTypes = VisitType::get();
        $customers = Customer::orderBy('created_at', 'desc')->get();

        return view("customers.create", compact('customers','visitTypes'));
    }

    public function store(Request $request)
    {


        $validator = Validator::make($request->all(), [
            "name" => "required|string|max:255",
            "phone" => "nullable|string|max:20",
        ], [
            "name.required" => __('customers.required_fields'),
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        if ($request->filled('customer_id')) {
            // تحديث الزبون
            $customer = Customer::find($request->customer_id);
        } else {
            // إنشاء زبون جديد
            $customer = Customer::create([
                "name" => $request->name,
                "phone" => $request->phone,
            ]);
        }

        // حساب أعلى ترتيب حالي وإضافة 1 له
        $maxPosition = WaitingQueue::whereDate('created_at', today())
    ->where('status', 'waiting')
    ->max('position') ?? 0;

        $newPosition = $maxPosition + 1;

        // إنشاء إدخال جديد في قائمة الانتظار
        WaitingQueue::create([
            "customer_id" => $customer->id,
            "type" => "walk-in",
            "visit_type" => $request->visit_type, // ← هنا
            "appointment_time" => $request->type == "appointment" ? $request->appointment_time : null,
            "status" => "waiting",
            "position" => $newPosition,
            "added_by" => Auth::id(),
        ]);

        // {{ __('activity_log.activity_log') }}
        ActivityLog::create([
            "user_id" => Auth::id(),
            "action" => "add_customer_to_queue",
            "description" => __('activity_log.customer_added_to_queue', ['name' => $customer->name, 'type' => $request->type]),
            "ip_address" => $request->ip(),
            "user_agent" => $request->userAgent(),
        ]);

         // Broadcast WebSocket update AFTER database changes are committed
        broadcast(new QueueUpdated(auth()->user()->tenant_id));

        return redirect()->route("dashboard")->with("success", __('customers.customer_added'));
    }
    public function show($id)
    {
        return response()->json(Customer::findOrFail($id));
    }

    public function destroy($id)
    {
        $customer = Customer::findOrFail($id);
        $customer->delete();

        return response()->json(['message' => __('customers.customer_deleted')]);
    }
}
